<?php

/**
 * StatusPage Renderer.
 *
 * Handles server-side rendering of status widgets.
 *
 * @package StatusPageWidget
 */

// Exit if accessed directly.
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Class StatusPage_Renderer
 *
 * Renders status widgets in PHP for server-side output.
 */
class StatusPage_Renderer
{

    /**
     * Status color mapping.
     *
     * @var array
     */
    private static $status_colors = array(
        'operational'    => '#16c172',
        'degraded'       => '#ffb400',
        'partial-outage' => '#ff7043',
        'major-outage'   => '#e53935',
        'maintenance'    => '#3870ff',
        'unknown'        => '#8c959f',
    );

    /**
     * Render a status widget.
     *
     * @param array $atts Widget attributes.
     * @return string Rendered HTML.
     */
    public static function render($atts)
    {
        $atts = wp_parse_args(
            $atts,
            array(
                'slug'           => '',
                'theme'          => 'auto',
                'compact'        => false,
                'show_incidents' => false,
                'rtm'            => 60,
                'upd'            => 7,
                'disable_link'   => false,
                'render_mode'    => 'js',
            )
        );

        // Sanitize attributes.
        $slug           = sanitize_text_field($atts['slug']);
        $theme          = in_array($atts['theme'], array('auto', 'light', 'dark'), true) ? $atts['theme'] : 'auto';
        $compact        = filter_var($atts['compact'], FILTER_VALIDATE_BOOLEAN);
        $show_incidents = filter_var($atts['show_incidents'], FILTER_VALIDATE_BOOLEAN);
        $rtm            = min(absint($atts['rtm']), 240);
        $upd            = min(absint($atts['upd']), 30);
        $disable_link   = filter_var($atts['disable_link'], FILTER_VALIDATE_BOOLEAN);
        $render_mode    = in_array($atts['render_mode'], array('js', 'php'), true) ? $atts['render_mode'] : 'js';

        if (empty($slug)) {
            return self::render_error(__('Please provide a status page slug.', 'status-page-widget'));
        }

        // Validate slug format.
        if (! self::is_valid_slug($slug)) {
            return self::render_error(__('Invalid slug format. Only lowercase letters, numbers, and single hyphens between characters are allowed.', 'status-page-widget'));
        }

        // JavaScript mode - output the embed script.
        if ('js' === $render_mode) {
            return self::render_js_embed($slug, $theme, $compact, $show_incidents, $rtm, $upd, $disable_link);
        }

        // PHP mode - fetch and render server-side.
        return self::render_php($slug, $theme, $compact, $show_incidents, $rtm, $upd, $disable_link);
    }

    /**
     * Render JavaScript embed mode.
     *
     * @param string $slug           Status page slug.
     * @param string $theme          Theme (auto, light, dark).
     * @param bool   $compact        Compact mode.
     * @param bool   $show_incidents Show incidents.
     * @param int    $rtm            Response time minutes.
     * @param int    $upd            Uptime days.
     * @param bool   $disable_link   Disable link.
     * @return string HTML output.
     */
    private static function render_js_embed($slug, $theme, $compact, $show_incidents, $rtm, $upd, $disable_link)
    {
        wp_enqueue_script('statuspage-embed');

        $data_attrs = array(
            'data-slug' => esc_attr($slug),
        );

        if ('auto' !== $theme) {
            $data_attrs['data-theme'] = esc_attr($theme);
        }

        if ($compact) {
            $data_attrs['data-compact'] = '1';
        }

        if ($show_incidents) {
            $data_attrs['data-show-incidents'] = '1';
        }

        if ($rtm > 0) {
            $data_attrs['data-rtm'] = esc_attr($rtm);
        }

        if ($upd > 0) {
            $data_attrs['data-upd'] = esc_attr($upd);
        }

        if ($disable_link) {
            $data_attrs['data-disable-link'] = '1';
        }

        $attrs_str = '';
        foreach ($data_attrs as $key => $value) {
            $attrs_str .= ' ' . $key . '="' . $value . '"';
        }

        return sprintf(
            '<script src="%s"%s></script>',
            esc_url(STATUSPAGE_API_BASE . '/static/embed/v1/status.min.js'),
            $attrs_str
        );
    }

    /**
     * Render PHP server-side mode.
     *
     * @param string $slug           Status page slug.
     * @param string $theme          Theme (auto, light, dark).
     * @param bool   $compact        Compact mode.
     * @param bool   $show_incidents Show incidents.
     * @param int    $rtm            Response time minutes.
     * @param int    $upd            Uptime days.
     * @param bool   $disable_link   Disable link.
     * @return string HTML output.
     */
    private static function render_php($slug, $theme, $compact, $show_incidents, $rtm, $upd, $disable_link)
    {
        // Fetch status data.
        $data = StatusPage_API::get_status($slug, $show_incidents ? 1 : 0, $rtm, $upd);

        if (is_wp_error($data)) {
            return self::render_error($data->get_error_message());
        }

        // Enqueue styles.
        wp_enqueue_style('status-page-widget-frontend');

        // Determine theme class.
        $theme_class = '';
        if ('dark' === $theme) {
            $theme_class = ' spw-dark';
        } elseif ('auto' === $theme) {
            $theme_class = ' spw-auto';
        }

        $compact_class = $compact ? ' spw-compact' : '';

        // Extract data.
        $overall   = isset($data['overall']) ? $data['overall'] : array(
            'code'  => 'unknown',
            'label' => __('Status Unknown', 'status-page-widget'),
        );
        $updated   = isset($data['updated']) ? $data['updated'] : '';
        $incidents = isset($data['incidents']) ? $data['incidents'] : array();
        $rtm_data  = isset($data['rtm']) ? $data['rtm'] : array();
        $upd_data  = isset($data['upd']) ? $data['upd'] : array();
        $dns_only  = isset($data['dns_only']) && $data['dns_only'];
        $page_url  = StatusPage_API::get_status_page_url($slug);

        // Start output.
        $html = '<div class="spw-wrap"><div class="spw' . esc_attr($theme_class . $compact_class) . '">';

        // Open link if not disabled.
        if (! $disable_link) {
            $html .= '<a href="' . esc_url($page_url) . '" target="_blank" rel="noopener" class="spw-link">';
        }

        // Header.
        $status_code = isset($overall['code']) ? $overall['code'] : 'unknown';
        $html       .= '<div class="spw-hd">';
        $html       .= '<span class="spw-dot spw-' . esc_attr($status_code) . '"></span>';
        $html       .= '<span class="spw-label">' . esc_html(isset($overall['label']) ? $overall['label'] : 'Status Unknown') . '</span>';

        // Compact sparkline.
        if ($compact && ! empty($rtm_data) && count($rtm_data) > 1 && ! $dns_only) {
            $spark_color = self::get_spark_color($status_code);
            $html       .= '<span class="spw-spark-wrap">' . self::generate_sparkline($rtm_data, 60, 14, $spark_color) . '</span>';
        }

        $html .= '</div>';

        // Meta (non-compact).
        if (! $compact && ! empty($updated)) {
            $updated_formatted = str_replace(array('T', 'Z'), array(' ', ' UTC'), $updated);
            $html             .= '<div class="spw-meta">' . esc_html(
                sprintf(
                    /* translators: %s: Last update time */
                    __('Updated %s', 'status-page-widget'),
                    $updated_formatted
                )
            ) . '</div>';
        }

        // Sparklines (non-compact).
        if (! $compact && ((! empty($rtm_data) && count($rtm_data) > 1 && ! $dns_only) || (! empty($upd_data) && count($upd_data) > 1))) {
            $html .= '<div class="spw-spark-wrap">';

            if (! empty($rtm_data) && count($rtm_data) > 1 && ! $dns_only) {
                $html .= '<div class="spw-spark" title="' . esc_attr(
                    sprintf(
                        /* translators: %d: Number of minutes */
                        __('Response time (ms, last %d min)', 'status-page-widget'),
                        count($rtm_data)
                    )
                ) . '">';
                $html .= self::generate_sparkline($rtm_data, 100, 24, '#16c172');
                $html .= '<div class="spw-spark-label">RT</div>';
                $html .= '</div>';
            }

            if (! empty($upd_data) && count($upd_data) > 1) {
                $html .= '<div class="spw-spark" title="' . esc_attr(
                    sprintf(
                        /* translators: %d: Number of days */
                        __('Uptime %% (last %d days)', 'status-page-widget'),
                        count($upd_data)
                    )
                ) . '">';
                $html .= self::generate_sparkline($upd_data, 100, 24, '#3870ff');
                $html .= '<div class="spw-spark-label">UP</div>';
                $html .= '</div>';
            }

            $html .= '</div>';
        }

        // Incidents (non-compact).
        if (! $compact && $show_incidents && ! empty($incidents)) {
            $html .= '<ul class="spw-incidents">';

            $display_incidents = array_slice($incidents, 0, 3);
            foreach ($display_incidents as $incident) {
                $html .= '<li>';
                $html .= '<a href="' . esc_url(STATUSPAGE_API_BASE . '/incident/' . $incident['id']) . '" target="_blank" rel="noopener">';
                $html .= esc_html($incident['title']);
                $html .= '</a>';

                if (! empty($incident['component'])) {
                    $html .= ' <span class="spw-muted">(' . esc_html($incident['component']) . ')</span>';
                }

                if (! empty($incident['state'])) {
                    $html .= ' <span class="spw-muted">(' . esc_html($incident['state']) . ')</span>';
                }

                $html .= '</li>';
            }

            if (count($incidents) > 3) {
                $html .= '<li class="spw-muted">' . esc_html(
                    sprintf(
                        /* translators: %d: Number of additional incidents */
                        __('+%d more', 'status-page-widget'),
                        count($incidents) - 3
                    )
                ) . '</li>';
            }

            $html .= '</ul>';
        }

        // Close link if not disabled.
        if (! $disable_link) {
            $html .= '</a>';
        }

        $html .= '</div></div>';

        return $html;
    }

    /**
     * Generate SVG sparkline.
     *
     * @param array  $numbers Array of numeric values.
     * @param int    $width   SVG width.
     * @param int    $height  SVG height.
     * @param string $stroke  Stroke color.
     * @return string SVG markup.
     */
    private static function generate_sparkline($numbers, $width, $height, $stroke)
    {
        if (empty($numbers) || count($numbers) < 2) {
            return '';
        }

        $len = count($numbers);
        $min = min($numbers);
        $max = max($numbers);

        if ($min === $max) {
            $max = $min + 1;
        }

        $dx  = $width / ($len - 1);
        $rng = $max - $min;

        // Build path.
        $y = $height - (($numbers[0] - $min) / $rng * $height);
        $d = 'M0 ' . number_format($y, 1, '.', '');

        for ($i = 1; $i < $len; $i++) {
            $y  = $height - (($numbers[$i] - $min) / $rng * $height);
            $d .= 'L' . number_format($dx * $i, 1, '.', '') . ' ' . number_format($y, 1, '.', '');
        }

        // Area path.
        $area = 'M0 ' . $height . ' ' . $d . ' L' . $width . ' ' . $height . ' Z';

        return sprintf(
            '<svg width="%d" height="%d" viewBox="0 0 %d %d"><path d="%s" fill="%s22"></path><path d="%s" fill="none" stroke="%s" stroke-width="1.2" stroke-linejoin="round" stroke-linecap="round"/></svg>',
            $width,
            $height,
            $width,
            $height,
            esc_attr($area),
            esc_attr($stroke),
            esc_attr($d),
            esc_attr($stroke)
        );
    }

    /**
     * Get sparkline color based on status.
     *
     * @param string $status_code Status code.
     * @return string Color hex code.
     */
    private static function get_spark_color($status_code)
    {
        if ('major-outage' === $status_code) {
            return '#e53935';
        } elseif ('degraded' === $status_code) {
            return '#ffb400';
        }
        return '#16c172';
    }

    /**
     * Render an error message.
     *
     * @param string $message Error message.
     * @return string HTML output.
     */
    private static function render_error($message)
    {
        return '<div class="spw-wrap"><div class="spw spw-error">'
            . '<div class="spw-hd">'
            . '<span class="spw-dot spw-unknown"></span>'
            . '<span class="spw-label">' . esc_html($message) . '</span>'
            . '</div>'
            . '</div></div>';
    }

    /**
     * Render an SVG badge.
     *
     * @param string $slug        Status page slug.
     * @param bool   $link        Link to status page.
     * @param string $alt         Alt text.
     * @return string HTML output.
     */
    public static function render_badge($slug, $link = true, $alt = '')
    {
        if (empty($slug)) {
            return '';
        }

        // Validate slug format.
        if (! self::is_valid_slug($slug)) {
            return '';
        }

        $badge_url = StatusPage_API::get_badge_url($slug);
        $page_url  = StatusPage_API::get_status_page_url($slug);
        $alt_text  = ! empty($alt) ? $alt : __('Status', 'status-page-widget');

        $img = sprintf(
            '<img src="%s" alt="%s" class="statuspage-badge" loading="lazy" />',
            esc_url($badge_url),
            esc_attr($alt_text)
        );

        if ($link) {
            return sprintf(
                '<a href="%s" target="_blank" rel="noopener" class="statuspage-badge-link">%s</a>',
                esc_url($page_url),
                $img
            );
        }

        return $img;
    }

    /**
     * Validate a slug format.
     *
     * Only allows lowercase letters (a-z), numbers (0-9), and single hyphens between characters.
     * Valid: nikola, ni-ko-la, nikola0-1-asd
     * Invalid: _nikola_, ---asdf_1, ni--ko-la
     *
     * @param string $slug The slug to validate.
     * @return bool Whether the slug is valid.
     */
    public static function is_valid_slug($slug)
    {
        if (empty($slug)) {
            return false;
        }

        return (bool) preg_match('/^[a-z0-9]+(-[a-z0-9]+)*$/', strtolower($slug));
    }
}
